/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.*;
import org.eclipse.swt.internal.*;
import org.eclipse.swt.events.*;

public abstract class Widget {
	int style, state;
	Display display;
	EventTable eventTable;
	Object data;
	
	Object jsObject;
	
	static final int DISPOSED		= 1<<0;
	static final int RELEASED		= 1<<1;
	static final int DISPOSE_SENT	= 1<<2;
	static final int KEYED_DATA		= 1<<3;

	static final int CANVAS			= 1<<4;
	static final int DISABLED		= 1<<5;
	static final int HIDDEN			= 1<<6;
	
	/* A layout was requested on this widget */
	static final int LAYOUT_NEEDED	= 1<<7;
	
	/* The preferred size of a child has changed */
	static final int LAYOUT_CHANGED = 1<<8;
	
	/* A layout was requested in this widget hierarchy */
	static final int LAYOUT_CHILD = 1<<9;
	
	static final int DRAG_DETECT = 1<<10;
	static final int FOREIGN_HANDLE = 1<<11;
	
	static final int MOVED = 1<<12;
	static final int RESIZED = 1<<13;
	
	static final int PARENT_BACKGROUND = 1<<14;
	
	/* Default size for widgets */
	static final int DEFAULT_WIDTH	= 64;
	static final int DEFAULT_HEIGHT	= 64;

Widget () {
	checkSubclass ();
}	

public Widget (Widget parent, int style) {
	this (parent, style, -1);
}

Widget (Widget parent, int style, int index) {
	createWidget (checkParent (parent), parent, style, index);
}

public void addDisposeListener (DisposeListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Dispose, typedListener);
}

void _addListener (int eventType, Listener listener) {
	if (eventTable == null) eventTable = new EventTable ();
	eventTable.hook (eventType, listener);
}

public void addListener (int eventType, Listener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	_addListener (eventType, listener);
}

static int checkBits (int style, int int0, int int1, int int2, int int3, int int4, int int5) {
	int mask = int0 | int1 | int2 | int3 | int4 | int5;
	if ((style & mask) == 0) style |= int0;
	if ((style & int0) != 0) style = (style & ~mask) | int0;
	if ((style & int1) != 0) style = (style & ~mask) | int1;
	if ((style & int2) != 0) style = (style & ~mask) | int2;
	if ((style & int3) != 0) style = (style & ~mask) | int3;
	if ((style & int4) != 0) style = (style & ~mask) | int4;
	if ((style & int5) != 0) style = (style & ~mask) | int5;
	return style;
}

boolean checkHandle (int handle) {
	return false;
}

void checkOpened () {
	/* Do nothing */
}

Display checkParent (Widget parent) {
	if (parent == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (parent.isDisposed ()) error (SWT.ERROR_INVALID_ARGUMENT);
	parent.checkWidget ();
	parent.checkOpened ();
	return parent.display;
}

protected void checkSubclass () {
	if (!isValidSubclass ()) error (SWT.ERROR_INVALID_SUBCLASS);
}

protected void checkWidget () {
	Display display = this.display;
	if (display == null) error (SWT.ERROR_WIDGET_DISPOSED);
	if (display.thread != Thread.currentThread ()) error (SWT.ERROR_THREAD_INVALID_ACCESS);
	if ((state & DISPOSED) != 0) error (SWT.ERROR_WIDGET_DISPOSED);
}

void createWidget (Display display, Widget parent, int style, int index) {
//TODO
//	checkSubclass ();
//	checkParent (parent);
	this.display = display;
	this.style = style;	
}

void destroyWidget () {	
	releaseHandle ();
}

public void dispose () {
	if (isDisposed ()) return;
	if (!isValidThread ()) error (SWT.ERROR_THREAD_INVALID_ACCESS);
	release (true);
}

void error (int code) {
	SWT.error(code);
}

public Object getData () {
	checkWidget();
	return (state & KEYED_DATA) != 0 ? ((Object []) data) [0] : data;
}

public Object getData (String key) {
	checkWidget();
	if (key == null) error (SWT.ERROR_NULL_ARGUMENT);
	if ((state & KEYED_DATA) != 0) {
		Object [] table = (Object []) data;
		for (int i=1; i<table.length; i+=2) {
			if (key.equals (table [i])) return table [i+1];
		}
	}
	return null;
}

public Display getDisplay () {
	Display display = this.display;
	if (display == null) error (SWT.ERROR_WIDGET_DISPOSED);
	return display;
}

Menu getMenu () {
	return null;
}

String getName () {
	//TODO: replace with proper code when getClass implemented
	//String string = getClass ().getName ();
	String string = "className";
	//
	int index = string.lastIndexOf ('.');
	if (index == -1) return string;
	return string.substring (index + 1, string.length ());
}

String getNameText () {
	return ""; //$NON-NLS-1$
}

//TODO uncomment if necessary
Object callMethod(String st1, String st2, Object[] obj){
	return null;
	/* empty */
}

Event getEvent(){
	Event event = new Event();
	event.display = display;
	event.widget = this;
	return event;
}

public Listener[] getListeners (int eventType) {
	checkWidget();
	if (eventTable == null) return new Listener[0];
	return eventTable.getListeners(eventType);
}

public int getStyle () {
	checkWidget();
	return style;
}

boolean hooks (int eventType) {
	if (eventTable == null) return false;
	return eventTable.hooks (eventType);
}

void hookEvents() {
}

void init () {
	_init();
}

public boolean isDisposed () {
	return (state & DISPOSED) != 0;
}

public boolean isListening (int eventType) {
	checkWidget();
	return hooks (eventType);
}

boolean isValidSubclass () {
	//TODO: add proper code when getClass added
	return true;
	//return Display.isValidClass (getClass ());
}

boolean isValidThread () {
//TODO implement	
//	return getDisplay ().isValidThread ();
	return true;
}

public void notifyListeners (int eventType, Event event) {
	checkWidget();
	if (event == null) event = new Event ();
	sendEvent (eventType, event);
}

void postEvent (int eventType) {
	sendEvent (eventType, null, false);
}

void postEvent (int eventType, Event event) {
	sendEvent (eventType, event, false);
}

void register () {	
}

void release (boolean destroy) {
	if ((state & DISPOSE_SENT) == 0) {
		state |= DISPOSE_SENT;
		sendEvent (SWT.Dispose);
	}
	if ((state & DISPOSED) == 0) {
		releaseChildren (destroy);
	}
	if ((state & RELEASED) == 0) {
		state |= RELEASED;
		if (destroy) {
			releaseParent ();
			releaseWidget ();
			destroyWidget ();
		} else {
			releaseWidget ();
			releaseHandle ();
		}
	}
}

void releaseChildren (boolean destroy) {
}

void releaseHandle () {
	state |= DISPOSED;
	display = null;
}

void releaseParent () {
}

void releaseWidget () {
	_releaseWidget();
	eventTable = null;
	data = null;
}

public void removeDisposeListener (DisposeListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Dispose, listener);
}

public void removeListener (int eventType, Listener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (eventType, listener);
}

protected void removeListener (int eventType, SWTEventListener listener) {
	checkWidget();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (eventType, listener);
}

void sendEvent (Event event) {
	Display display = event.display;
	if (!display.filterEvent (event)) {
		if (eventTable != null) eventTable.sendEvent (event);
	}
}

void sendEvent (int eventType) {
	sendEvent (eventType, null, true);
}

void sendEvent (int eventType, Event event) {
	sendEvent (eventType, event, true);
}

void sendEvent (int eventType, Event event, boolean send) {
	if (eventTable == null && !display.filters (eventType)) {
		return;
	}
	if (event == null) event = new Event ();
	event.type = eventType;
	event.display = display;
	event.widget = this;
	if (event.time == 0) {
		event.time = display.getLastEventTime ();
	}
	if (send) {
		sendEvent (event);
	} else {
		display.postEvent (event);
	}
}

public void setData (Object data) {
	checkWidget();
	if ((state & KEYED_DATA) != 0) {
		((Object []) this.data) [0] = data;
	} else {
		this.data = data;
	}
}

public void setData (String key, Object value) {
	checkWidget();
	if (key == null) error (SWT.ERROR_NULL_ARGUMENT);
	int index = 1;
	Object [] table = null;
	if ((state & KEYED_DATA) != 0) {
		table = (Object []) data;
		while (index < table.length) {
			if (key.equals (table [index])) break;
			index += 2;
		}
	}
	if (value != null) {
		if ((state & KEYED_DATA) != 0) {
			if (index == table.length) {
				Object [] newTable = new Object [table.length + 2];
				System.arraycopy (table, 0, newTable, 0, table.length);
				data = table = newTable;
			}
		} else {
			table = new Object [3];
			table [0] = data;
			data = table;
			state |= KEYED_DATA;
		}
		table [index] = key;
		table [index + 1] = value;
	} else {
		if ((state & KEYED_DATA) != 0) {
			if (index != table.length) {
				int length = table.length - 2;
				if (length == 1) {
					data = table [0];
					state &= ~KEYED_DATA;
				} else {
					Object [] newTable = new Object [length];
					System.arraycopy (table, 0, newTable, 0, index);
					System.arraycopy (table, index + 2, newTable, index, length - index);
					data = newTable;
				}
			}
		}
	}
}

boolean showMenu (int x, int y) {
	Event event = new Event ();
	event.x = x;
	event.y = y;
	sendEvent (SWT.MenuDetect, event);
	if (!event.doit) return true;
	Menu menu = getMenu ();
	if (menu != null && !menu.isDisposed ()) {
//		if (x != event.x || y != event.y) {
		menu.setLocation (event.x, event.y);
//		}
		menu.setVisible (true);
		return true;
	}
	return false;
}

public String toString () {
	String string = "*Disposed*"; //$NON-NLS-1$
	if (!isDisposed ()) {
		string = "*Wrong Thread*"; //$NON-NLS-1$
		if (isValidThread ()) string = getNameText ();
	}
	return getName () + " {" + string + "}"; //$NON-NLS-1$ //$NON-NLS-2$
}

/*---------------------- COMMON INTERFACE -------------------*/

native void _hookEvents (String eventType, int style) /*-{
	var self = this;
	$wnd.dojo.connect(
		self.@org.eclipse.swt.widgets.Widget::jsObject.domNode,
		eventType, 
		function(arg){
			if (arg) {
//				arg.preventDefault();
				var event = self.@org.eclipse.swt.widgets.Widget::getEvent()();
				event.@org.eclipse.swt.widgets.Event::type = style;
				event.@org.eclipse.swt.widgets.Event::x = arg.pageX;
				event.@org.eclipse.swt.widgets.Event::y = arg.pageY;
				self.@org.eclipse.swt.widgets.Widget::sendEvent(ILorg/eclipse/swt/widgets/Event;)(style, event);
			}
		}
	);
}-*/;

native void _hookCalls (String jsMethodName, int style) /*-{
	var self = this;
	$wnd.dojo.connect(
		self.@org.eclipse.swt.widgets.Widget::jsObject,
		jsMethodName,
		function(arg){
			if (arg) {
				var event = self.@org.eclipse.swt.widgets.Widget::getEvent()();
				event.@org.eclipse.swt.widgets.Event::type = style;
				event.@org.eclipse.swt.widgets.Event::x = arg.pageX;
				event.@org.eclipse.swt.widgets.Event::y = arg.pageY;
				self.@org.eclipse.swt.widgets.Widget::sendEvent(ILorg/eclipse/swt/widgets/Event;)(style, event);
			}
		}
	);
}-*/;

native void _init () /*-{
//	if (this.@org.eclipse.swt.widgets.Widget::jsObject == null) {
//		this.@org.eclipse.swt.widgets.Widget::jsObject = $doc.createElement("div");
//	}
}-*/;

native void _releaseWidget () /*-{
	if (this.@org.eclipse.swt.widgets.Widget::jsObject) {
		if (this.@org.eclipse.swt.widgets.Widget::jsObject.destroy) {
			this.@org.eclipse.swt.widgets.Widget::jsObject.destroy();
		}
	}
	this.@org.eclipse.swt.widgets.Widget::jsObject = null;
}-*/;

}
